import { Link, useLocation } from "wouter";
import { cn } from "@/lib/utils";
import { Book, Search, Calculator, GitCompare, Menu, FileText, Droplets } from "lucide-react";
import { Sheet, SheetContent, SheetTrigger } from "@/components/ui/sheet";
import { Button } from "@/components/ui/button";
import vintagePaper from "@assets/generated_images/vintage_off-white_paper_texture_background.png";

interface LayoutProps {
  children: React.ReactNode;
}

export function Layout({ children }: LayoutProps) {
  const [location] = useLocation();

  const navItems = [
    { href: "/", label: "Home", icon: Book },
    { href: "/search", label: "Part Look-Up", icon: Search },
    { href: "/compare", label: "Compare Specs", icon: GitCompare },
    { href: "/calculator", label: "HP Calculator", icon: Calculator },
    { href: "/tech-reference", label: "Tech Reference", icon: FileText },
    { href: "/lubes-n-things", label: "Lubes n Things", icon: Droplets },
  ];

  const isActive = (path: string) => location === path;

  return (
    <div 
      className="min-h-screen bg-background font-sans text-foreground flex flex-col"
      style={{
        backgroundImage: `url(${vintagePaper})`,
        backgroundSize: "cover",
        backgroundBlendMode: "multiply",
      }}
    >
      {/* Header */}
      <header className="border-b-4 border-primary bg-background/90 backdrop-blur-sm sticky top-0 z-50 shadow-md">
        <div className="container mx-auto px-4 h-16 flex items-center justify-between">
          <div className="flex items-center gap-3">
            <div className="w-10 h-10 bg-primary text-primary-foreground flex items-center justify-center rounded-sm font-serif font-bold text-xl border-2 border-primary-foreground shadow-sm">
              GM
            </div>
            <div className="flex flex-col">
              <span className="font-serif font-bold text-lg leading-tight text-primary tracking-wide">
                SMALL BLOCK
              </span>
              <span className="font-mono text-xs uppercase tracking-widest text-muted-foreground">
                Code Book & Reference
              </span>
            </div>
          </div>

          {/* Desktop Nav */}
          <nav className="hidden md:flex items-center gap-1">
            {navItems.map((item) => (
              <Link 
                key={item.href} 
                href={item.href}
                className={cn(
                  "flex items-center gap-2 px-4 py-2 rounded-md transition-all font-medium text-sm",
                  isActive(item.href)
                    ? "bg-primary text-primary-foreground shadow-md"
                    : "text-foreground/70 hover:bg-primary/10 hover:text-primary"
                )}
              >
                <item.icon className="w-4 h-4" />
                {item.label}
              </Link>
            ))}
            
          </nav>

          {/* Mobile Nav */}
          <Sheet>
            <SheetTrigger asChild>
              <Button variant="ghost" size="icon" className="md:hidden">
                <Menu className="w-6 h-6" />
              </Button>
            </SheetTrigger>
            <SheetContent side="right" className="bg-[#f4f1ea] border-l-4 border-primary">
              <div className="flex flex-col gap-6 mt-10">
                {navItems.map((item) => (
                  <Link 
                    key={item.href} 
                    href={item.href}
                    className={cn(
                      "flex items-center gap-4 px-4 py-3 rounded-md transition-all font-serif font-bold text-lg",
                      isActive(item.href)
                        ? "bg-primary text-primary-foreground shadow-md"
                        : "text-foreground/70 hover:bg-primary/10 hover:text-primary"
                    )}
                  >
                    <item.icon className="w-5 h-5" />
                    {item.label}
                  </Link>
                ))}
              </div>
            </SheetContent>
          </Sheet>
        </div>
      </header>

      {/* Main Content */}
      <main className="flex-1 container mx-auto px-4 py-8 relative">
        {/* Decorative Grid Lines */}
        <div className="absolute inset-0 pointer-events-none opacity-[0.03]" 
          style={{ 
            backgroundImage: "linear-gradient(#003366 1px, transparent 1px), linear-gradient(90deg, #003366 1px, transparent 1px)", 
            backgroundSize: "20px 20px" 
          }} 
        />
        <div className="relative z-10">
          {children}
        </div>
      </main>

      {/* Footer */}
      <footer className="bg-foreground text-background py-8 border-t-4 border-secondary mt-auto">
        <div className="container mx-auto px-4 text-center">
          <div className="flex items-center justify-center gap-2 mb-4 opacity-50">
            <div className="w-8 h-8 border border-current rounded-full flex items-center justify-center font-serif font-bold text-xs">LS</div>
            <span className="font-mono text-xs uppercase tracking-widest">Independent Reference Guide</span>
          </div>
          
          {/* Legal Disclaimer */}
          <div className="max-w-2xl mx-auto mb-4 p-4 bg-background/10 rounded border border-background/20">
            <p className="text-xs font-medium uppercase tracking-wider mb-2 opacity-80">Legal Disclaimer</p>
            <p className="text-xs opacity-70 leading-relaxed">
              This website is <strong>NOT affiliated with, endorsed by, or certified by General Motors Corporation</strong> or any of its subsidiaries. 
              All trademarks, brand names, and part numbers referenced herein are the property of their respective owners. 
              This site is an <strong>independent research tool</strong> created for educational and informational purposes only. 
              Information provided may contain errors or inaccuracies. Always verify critical specifications with official GM documentation or a qualified professional.
            </p>
          </div>
          
          <p className="text-xs opacity-40">
            &copy; {new Date().getFullYear()} LS Engine Reference Guide. For educational and restoration research purposes only.
          </p>
        </div>
      </footer>
    </div>
  );
}
