import { Part } from "@/lib/gm-data";
import { Badge } from "@/components/ui/badge";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { ArrowRight, Box, Cylinder, Settings, Layers, Cog, Zap } from "lucide-react";
import { Link } from "wouter";
import { PartsStoreButton } from "./PartsStoreButton";

interface PartCardProps {
  part: Part;
  compact?: boolean;
}

export function PartCard({ part, compact = false }: PartCardProps) {
  const getIcon = () => {
    switch (part.category) {
      case 'Block': return Box;
      case 'Head': return Cylinder;
      case 'Camshaft': return Settings;
      case 'Intake': return Layers;
      case 'Injector': return Zap;
      default: return Cog;
    }
  };

  const Icon = getIcon();

  return (
    <Link href={`/search?part=${part.partNumber}`} className="block h-full group">
      <Card className="h-full border-2 border-border/50 hover:border-primary transition-all duration-300 hover:shadow-lg bg-card/80 backdrop-blur-sm group-hover:-translate-y-1">
        <CardHeader className="pb-2 space-y-1">
          <div className="flex justify-between items-start">
            <Badge variant="outline" className="font-mono text-xs bg-secondary/50 text-secondary-foreground border-border/50">
              {part.generation}
            </Badge>
            <div className="w-8 h-8 bg-primary/10 rounded-full flex items-center justify-center text-primary group-hover:bg-primary group-hover:text-primary-foreground transition-colors">
              <Icon className="w-4 h-4" />
            </div>
          </div>
          <CardTitle className="font-serif text-lg leading-tight group-hover:text-primary transition-colors">
            {part.name}
          </CardTitle>
          <div className="font-mono text-xs text-muted-foreground flex items-center gap-1">
            {part.category === 'Camshaft' 
              ? (part.stampedCode ? `Stamp: ${part.stampedCode}` : 'Part #:') 
              : 'Casting #:'} 
            <span className="text-foreground font-bold">{part.partNumber}</span>
          </div>
          <div className="flex gap-2 mt-2">
            <PartsStoreButton partNumber={part.partNumber} store="Summit" />
          </div>
        </CardHeader>
        {!compact && (
          <CardContent>
            <p className="text-sm text-muted-foreground line-clamp-2 mb-4 font-medium leading-relaxed">
              {part.description}
            </p>
            <div className="grid grid-cols-2 gap-2">
              {Object.entries(part.specs).map(([key, value]) => (
                <div key={key} className="bg-secondary/30 p-2 rounded border border-border/30">
                  <div className="text-[10px] uppercase tracking-wider text-muted-foreground font-bold">{key}</div>
                  <div className="font-mono text-xs font-semibold">{value}</div>
                </div>
              ))}
            </div>
            <div className="mt-4 flex items-center text-primary text-xs font-bold uppercase tracking-widest group-hover:underline">
              View Specs <ArrowRight className="w-3 h-3 ml-1" />
            </div>
          </CardContent>
        )}
      </Card>
    </Link>
  );
}
