import { Layout } from "@/components/Layout";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { gmParts, Part } from "@/lib/gm-data";
import { useState } from "react";
import { AlertCircle, ArrowRightLeft } from "lucide-react";
import { Alert, AlertDescription, AlertTitle } from "@/components/ui/alert";

export default function Compare() {
  const [partAId, setPartAId] = useState<string>("");
  const [partBId, setPartBId] = useState<string>("");

  const partA = gmParts.find(p => p.id === partAId);
  const partB = gmParts.find(p => p.id === partBId);

  // Group parts by category for the select dropdown
  const categories = Array.from(new Set(gmParts.map(p => p.category)));

  // Get all unique spec keys from both selected parts
  const allSpecKeys = Array.from(new Set([
    ...(partA ? Object.keys(partA.specs) : []),
    ...(partB ? Object.keys(partB.specs) : [])
  ]));

  return (
    <Layout>
      <div className="space-y-8">
        <div className="border-b pb-6 border-border/50">
          <h1 className="text-4xl font-serif font-bold mb-2">Specification Comparator</h1>
          <p className="text-muted-foreground font-mono text-sm">
            Select two components to analyze differences side-by-side.
          </p>
        </div>

        {/* Selection Area */}
        <div className="grid grid-cols-1 md:grid-cols-[1fr_auto_1fr] gap-6 items-center">
          {/* Part A Selector */}
          <Card className="border-2 border-primary/20 shadow-lg">
            <CardHeader>
              <CardTitle className="text-center font-serif text-primary">Component A</CardTitle>
            </CardHeader>
            <CardContent>
              <Select value={partAId} onValueChange={setPartAId}>
                <SelectTrigger className="w-full">
                  <SelectValue placeholder="Select Part..." />
                </SelectTrigger>
                <SelectContent className="max-h-[300px] overflow-y-auto">
                  {categories.map(cat => (
                    <div key={cat}>
                      <div className="px-2 py-1.5 text-xs font-bold text-muted-foreground bg-muted/50 uppercase tracking-wider sticky top-0">
                        {cat}
                      </div>
                      {gmParts.filter(p => p.category === cat).map(p => (
                        <SelectItem key={p.id} value={p.id} className="font-mono text-xs">
                          {p.partNumber} - {p.name}
                        </SelectItem>
                      ))}
                    </div>
                  ))}
                </SelectContent>
              </Select>
            </CardContent>
          </Card>

          <div className="flex justify-center">
            <div className="w-12 h-12 rounded-full bg-primary text-primary-foreground flex items-center justify-center shadow-xl">
              <ArrowRightLeft className="w-6 h-6" />
            </div>
          </div>

          {/* Part B Selector */}
          <Card className="border-2 border-primary/20 shadow-lg">
            <CardHeader>
              <CardTitle className="text-center font-serif text-primary">Component B</CardTitle>
            </CardHeader>
            <CardContent>
              <Select value={partBId} onValueChange={setPartBId}>
                <SelectTrigger className="w-full">
                  <SelectValue placeholder="Select Part..." />
                </SelectTrigger>
                <SelectContent className="max-h-[300px] overflow-y-auto">
                  {categories.map(cat => (
                    <div key={cat}>
                      <div className="px-2 py-1.5 text-xs font-bold text-muted-foreground bg-muted/50 uppercase tracking-wider sticky top-0">
                        {cat}
                      </div>
                      {gmParts.filter(p => p.category === cat).map(p => (
                        <SelectItem key={p.id} value={p.id} className="font-mono text-xs">
                          {p.partNumber} - {p.name}
                        </SelectItem>
                      ))}
                    </div>
                  ))}
                </SelectContent>
              </Select>
            </CardContent>
          </Card>
        </div>

        {/* Comparison Table */}
        {partA && partB ? (
          <Card className="animate-in fade-in slide-in-from-bottom-8 duration-500 border-2 border-primary/10">
            <CardHeader>
              <CardTitle className="text-center">Technical Specification Analysis</CardTitle>
            </CardHeader>
            <CardContent>
              {partA.category !== partB.category && (
                <Alert variant="destructive" className="mb-6">
                  <AlertCircle className="h-4 w-4" />
                  <AlertTitle>Mismatch Warning</AlertTitle>
                  <AlertDescription>
                    You are comparing a <strong>{partA.category}</strong> with a <strong>{partB.category}</strong>. 
                    Specifications may not directly align.
                  </AlertDescription>
                </Alert>
              )}

              <div className="rounded-md border border-border overflow-hidden">
                <Table>
                  <TableHeader className="bg-muted/30">
                    <TableRow>
                      <TableHead className="w-[200px] font-bold">Specification</TableHead>
                      <TableHead className="font-bold text-primary">{partA.name}</TableHead>
                      <TableHead className="font-bold text-primary">{partB.name}</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {/* Basic Info Rows */}
                    <TableRow>
                      <TableCell className="font-medium text-muted-foreground">Part Number</TableCell>
                      <TableCell className="font-mono font-bold">{partA.partNumber}</TableCell>
                      <TableCell className="font-mono font-bold">{partB.partNumber}</TableCell>
                    </TableRow>
                    <TableRow>
                      <TableCell className="font-medium text-muted-foreground">Generation</TableCell>
                      <TableCell>{partA.generation}</TableCell>
                      <TableCell>{partB.generation}</TableCell>
                    </TableRow>
                    <TableRow>
                      <TableCell className="font-medium text-muted-foreground">Description</TableCell>
                      <TableCell className="text-xs">{partA.description}</TableCell>
                      <TableCell className="text-xs">{partB.description}</TableCell>
                    </TableRow>
                    
                    {/* Dynamic Specs Rows */}
                    {allSpecKeys.map(key => (
                      <TableRow key={key}>
                        <TableCell className="font-medium text-muted-foreground bg-muted/10">{key}</TableCell>
                        <TableCell className="font-mono">
                          {partA.specs[key] || <span className="text-muted-foreground/30 italic">N/A</span>}
                        </TableCell>
                        <TableCell className="font-mono">
                          {partB.specs[key] || <span className="text-muted-foreground/30 italic">N/A</span>}
                        </TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              </div>
            </CardContent>
          </Card>
        ) : (
          <div className="text-center py-20 opacity-50 border-2 border-dashed border-border rounded-lg">
            <ArrowRightLeft className="w-12 h-12 mx-auto mb-4" />
            <h3 className="text-xl font-serif">Awaiting Selection</h3>
            <p>Select two components above to begin comparison.</p>
          </div>
        )}
      </div>
    </Layout>
  );
}
