import { Layout } from "@/components/Layout";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Search, ArrowRight, Settings, Info, Beer } from "lucide-react";
import { gmParts } from "@/lib/gm-data";
import { PartCard } from "@/components/PartCard";
import { Link, useLocation } from "wouter";
import { useState } from "react";
import { Dialog, DialogContent, DialogHeader, DialogTitle } from "@/components/ui/dialog";

export default function Home() {
  const [searchQuery, setSearchQuery] = useState("");
  const [, setLocation] = useLocation();
  const [aboutOpen, setAboutOpen] = useState(false);

  const handleSearch = (e: React.FormEvent) => {
    e.preventDefault();
    if (searchQuery.trim()) {
      setLocation(`/search?q=${encodeURIComponent(searchQuery)}`);
    }
  };

  const featuredParts = gmParts.slice(0, 3); // Just show first 3 for demo

  return (
    <Layout>
      {/* Hero Section */}
      <section className="py-16 md:py-24 text-center space-y-8 max-w-4xl mx-auto">
        <div className="space-y-4 animate-in fade-in slide-in-from-bottom-4 duration-700">
          <div className="inline-flex items-center gap-2 px-3 py-1 rounded-full bg-primary/10 text-primary border border-primary/20 text-xs font-bold uppercase tracking-widest mb-4">
            <Info className="w-3 h-3" /> The Definitive Resource
          </div>
          <h1 className="text-5xl md:text-7xl font-serif font-black text-foreground tracking-tight leading-[0.9]">
            SMALL BLOCK <br/>
            <span className="text-primary text-transparent bg-clip-text bg-gradient-to-r from-primary to-blue-600">
              KNOWLEDGE
            </span>
          </h1>
          <p className="text-lg md:text-xl text-muted-foreground font-medium max-w-2xl mx-auto leading-relaxed">
            The comprehensive digital handbook for GM Gen 3 & Gen 4 LS-platform engines. 
            Identify parts, compare specifications, and calculate potential.
          </p>
        </div>

        {/* Search Box */}
        <div className="max-w-xl mx-auto relative z-20 animate-in fade-in slide-in-from-bottom-8 duration-700 delay-100">
          <form onSubmit={handleSearch} className="relative group">
            <div className="absolute inset-0 bg-primary/20 blur-xl rounded-full group-hover:bg-primary/30 transition-all opacity-0 group-hover:opacity-100" />
            <div className="relative flex shadow-xl">
              <Input 
                type="text" 
                placeholder="Search by Part Number (e.g., 12561168) or Name..." 
                className="h-14 px-6 rounded-l-md border-2 border-r-0 border-primary/20 focus-visible:ring-0 focus-visible:border-primary bg-background font-mono text-lg"
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
              />
              <Button type="submit" size="lg" className="h-14 px-8 rounded-l-none rounded-r-md font-bold text-lg">
                <Search className="w-5 h-5 mr-2" />
                Look Up
              </Button>
            </div>
          </form>
          <div className="mt-4 flex justify-center gap-4 text-sm text-muted-foreground font-mono">
            <span>Try:</span>
            <button onClick={() => setLocation('/search?q=LS3')} className="hover:text-primary underline decoration-dotted">LS3 Block</button>
            <button onClick={() => setLocation('/search?q=243')} className="hover:text-primary underline decoration-dotted">243 Heads</button>
            <button onClick={() => setLocation('/search?q=Cam')} className="hover:text-primary underline decoration-dotted">Camshafts</button>
          </div>
        </div>

        {/* Buy Me a Beer Button */}
        <div className="mt-8 animate-in fade-in slide-in-from-bottom-10 duration-700 delay-150">
          <Button 
            size="lg"
            className="bg-amber-500 hover:bg-amber-600 text-black font-bold text-lg px-8 py-6 shadow-lg hover:shadow-xl transition-all"
            onClick={() => setAboutOpen(true)}
            data-testid="button-buy-beer-home"
          >
            <Beer className="w-6 h-6 mr-3" />
            Buy Me a Beer!
          </Button>
        </div>
      </section>

      {/* Quick Actions */}
      <section className="grid grid-cols-1 md:grid-cols-3 gap-6 mb-20 animate-in fade-in slide-in-from-bottom-12 duration-700 delay-200">
        <Link href="/search" className="group p-8 rounded-lg border-2 border-border/50 hover:border-primary bg-card/50 hover:bg-card transition-all text-center space-y-4 hover:shadow-xl">
          <div className="w-16 h-16 mx-auto bg-primary/10 rounded-full flex items-center justify-center text-primary group-hover:scale-110 transition-transform">
            <Search className="w-8 h-8" />
          </div>
          <h3 className="text-2xl font-serif font-bold">Part Identification</h3>
          <p className="text-muted-foreground">Decode part numbers and view detailed factory specifications.</p>
        </Link>
        <Link href="/compare" className="group p-8 rounded-lg border-2 border-border/50 hover:border-primary bg-card/50 hover:bg-card transition-all text-center space-y-4 hover:shadow-xl">
          <div className="w-16 h-16 mx-auto bg-primary/10 rounded-full flex items-center justify-center text-primary group-hover:scale-110 transition-transform">
            <Settings className="w-8 h-8" />
          </div>
          <h3 className="text-2xl font-serif font-bold">Spec Comparison</h3>
          <p className="text-muted-foreground">Side-by-side comparison of heads, cams, and blocks.</p>
        </Link>
        <Link href="/calculator" className="group p-8 rounded-lg border-2 border-border/50 hover:border-primary bg-card/50 hover:bg-card transition-all text-center space-y-4 hover:shadow-xl">
          <div className="w-16 h-16 mx-auto bg-primary/10 rounded-full flex items-center justify-center text-primary group-hover:scale-110 transition-transform">
            <ArrowRight className="w-8 h-8" />
          </div>
          <h3 className="text-2xl font-serif font-bold">HP Calculator</h3>
          <p className="text-muted-foreground">Theoretical power estimates based on component combinations.</p>
        </Link>
      </section>

      {/* Featured Parts */}
      <section className="space-y-8 animate-in fade-in slide-in-from-bottom-16 duration-700 delay-300">
        <div className="flex items-center justify-between border-b-2 border-border/50 pb-4">
          <h2 className="text-3xl font-serif font-bold text-foreground">Featured Components</h2>
          <Link href="/search" className="text-primary font-bold uppercase tracking-widest text-sm hover:underline flex items-center gap-1">
            View All <ArrowRight className="w-4 h-4" />
          </Link>
        </div>
        <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
          {featuredParts.map(part => (
            <PartCard key={part.id} part={part} />
          ))}
        </div>
      </section>

      {/* About Developer Modal */}
      <Dialog open={aboutOpen} onOpenChange={setAboutOpen}>
        <DialogContent className="bg-[#1e1e1e] text-white border-none max-w-md">
          <DialogHeader>
            <DialogTitle className="text-2xl font-serif font-bold text-white">About This App</DialogTitle>
          </DialogHeader>
          
          <div className="max-h-[300px] overflow-y-auto pr-2 mt-4">
            <p className="text-gray-300 leading-relaxed text-sm">
              I built this app because I was tired of digging through forums to find reliable LS part numbers. 
              I wanted a clean, no-nonsense tool for the shop. If this app saved you time on your build, 
              consider buying me a beer to keep the updates coming.
            </p>
          </div>
          
          <div className="mt-6 pt-4 border-t border-gray-700">
            <Button 
              className="w-full bg-amber-500 hover:bg-amber-600 text-black font-bold py-3"
              onClick={() => window.open('https://ko-fi.com', '_blank')}
              data-testid="button-buy-beer-modal"
            >
              <Beer className="w-5 h-5 mr-2" />
              Buy Me a Beer
            </Button>
          </div>
        </DialogContent>
      </Dialog>
    </Layout>
  );
}
