import { Layout } from "@/components/Layout";
import { Input } from "@/components/ui/input";
import { Button } from "@/components/ui/button";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { PartCard } from "@/components/PartCard";
import { gmParts, Part } from "@/lib/gm-data";
import { Search as SearchIcon, Filter, X } from "lucide-react";
import { useState, useEffect } from "react";
import { useLocation } from "wouter";

export default function Search() {
  const [location] = useLocation();
  const [query, setQuery] = useState("");
  const [categoryFilter, setCategoryFilter] = useState<string>("all");
  const [genFilter, setGenFilter] = useState<string>("all");
  const [filteredParts, setFilteredParts] = useState<Part[]>(gmParts);

  // Parse query param on mount
  useEffect(() => {
    const params = new URLSearchParams(window.location.search);
    const q = params.get("q");
    const part = params.get("part");
    if (q) setQuery(q);
    if (part) setQuery(part);
  }, []);

  // Filter logic
  useEffect(() => {
    let results = gmParts;

    if (query) {
      const lowerQ = query.toLowerCase();
      results = results.filter(p => 
        p.name.toLowerCase().includes(lowerQ) || 
        p.partNumber.includes(lowerQ) ||
        p.description.toLowerCase().includes(lowerQ) ||
        (p.stampedCode && p.stampedCode.includes(lowerQ)) ||
        Object.values(p.specs).some(val => String(val).toLowerCase().includes(lowerQ))
      );
    }

    if (categoryFilter !== "all") {
      results = results.filter(p => p.category === categoryFilter);
    }

    if (genFilter !== "all") {
      results = results.filter(p => p.generation === genFilter);
    }

    setFilteredParts(results);
  }, [query, categoryFilter, genFilter]);

  const clearFilters = () => {
    setQuery("");
    setCategoryFilter("all");
    setGenFilter("all");
  };

  return (
    <Layout>
      <div className="space-y-8">
        <div className="flex flex-col md:flex-row justify-between items-start md:items-center gap-4 border-b pb-6 border-border/50">
          <div>
            <h1 className="text-4xl font-serif font-bold mb-2">Part Catalog</h1>
            <p className="text-muted-foreground font-mono text-sm">
              Indexing {gmParts.length} verified GM performance components
            </p>
          </div>
          
          <div className="flex gap-2 w-full md:w-auto">
            <Button variant="outline" size="sm" onClick={clearFilters} disabled={!query && categoryFilter === "all" && genFilter === "all"}>
              <X className="w-4 h-4 mr-2" /> Clear
            </Button>
          </div>
        </div>

        {/* Search & Filter Bar */}
        <div className="grid grid-cols-1 md:grid-cols-12 gap-4 bg-card p-4 rounded-lg border-2 border-border/50 shadow-sm">
          <div className="md:col-span-6 relative">
            <SearchIcon className="absolute left-3 top-1/2 -translate-y-1/2 text-muted-foreground w-5 h-5" />
            <Input 
              placeholder="Search by Part # or Keyword..." 
              className="pl-10 h-10 font-mono"
              value={query}
              onChange={(e) => setQuery(e.target.value)}
            />
          </div>
          
          <div className="md:col-span-3">
            <Select value={categoryFilter} onValueChange={setCategoryFilter}>
              <SelectTrigger className="h-10">
                <Filter className="w-4 h-4 mr-2 text-muted-foreground" />
                <SelectValue placeholder="Category" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all">All Categories</SelectItem>
                <SelectItem value="Block">Blocks</SelectItem>
                <SelectItem value="Head">Heads</SelectItem>
                <SelectItem value="Camshaft">Camshafts</SelectItem>
                <SelectItem value="Intake">Intakes</SelectItem>
                <SelectItem value="Injector">Injectors</SelectItem>
                <SelectItem value="Oil Pump">Oil Pumps</SelectItem>
                <SelectItem value="Oil Pan">Oil Pans</SelectItem>
              </SelectContent>
            </Select>
          </div>

          <div className="md:col-span-3">
            <Select value={genFilter} onValueChange={setGenFilter}>
              <SelectTrigger className="h-10">
                <SelectValue placeholder="Generation" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all">All Generations</SelectItem>
                <SelectItem value="Gen 3">Gen 3 (LS1/LS6/LQ4)</SelectItem>
                <SelectItem value="Gen 4">Gen 4 (LS2/LS3/LS7)</SelectItem>
              </SelectContent>
            </Select>
          </div>
        </div>

        {/* Results */}
        {filteredParts.length > 0 ? (
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6 animate-in fade-in slide-in-from-bottom-4 duration-500">
            {filteredParts.map(part => (
              <PartCard key={part.id} part={part} />
            ))}
          </div>
        ) : (
          <div className="text-center py-20 bg-card/30 rounded-lg border-2 border-dashed border-border">
            <SearchIcon className="w-12 h-12 mx-auto text-muted-foreground mb-4 opacity-50" />
            <h3 className="text-xl font-bold font-serif mb-2">No Parts Found</h3>
            <p className="text-muted-foreground">Try adjusting your search filters or check the part number.</p>
            <Button variant="link" onClick={clearFilters} className="mt-4">Reset Search</Button>
          </div>
        )}
      </div>
    </Layout>
  );
}
