import { Layout } from "@/components/Layout";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { AlertTriangle, CheckCircle2, Info, XCircle } from "lucide-react";

export default function TechReference() {
  return (
    <Layout>
      <div className="max-w-4xl mx-auto space-y-12">
        {/* Header */}
        <div className="text-center space-y-4 border-b-2 border-primary/20 pb-8">
          <Badge variant="outline" className="font-mono text-primary border-primary">TECHNICAL BULLETIN 001</Badge>
          <h1 className="text-4xl md:text-5xl font-serif font-bold text-foreground">
            Gen 3 vs. Gen 4 Architecture
          </h1>
          <p className="text-xl text-muted-foreground font-medium max-w-2xl mx-auto">
            A definitive guide to identifying differences, interchangeability, and the evolution of the LS platform.
          </p>
        </div>

        {/* Quick ID Matrix */}
        <section className="space-y-6">
          <div className="flex items-center gap-2 mb-4">
            <div className="w-8 h-8 bg-primary text-primary-foreground flex items-center justify-center rounded-sm font-bold font-serif">1</div>
            <h2 className="text-2xl font-serif font-bold">Quick Identification Guide</h2>
          </div>
          
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            <Card className="border-2 border-border/60 bg-card/50">
              <CardHeader className="bg-muted/20 border-b border-border/20">
                <CardTitle className="font-mono text-lg flex justify-between items-center">
                  GENERATION 3
                  <Badge variant="secondary">1997 - 2007</Badge>
                </CardTitle>
              </CardHeader>
              <CardContent className="pt-6 space-y-4">
                <ul className="space-y-3 text-sm">
                  <li className="flex gap-3">
                    <span className="font-bold min-w-[120px]">Reluctor Wheel:</span>
                    <span>24x (Black Crank Sensor)</span>
                  </li>
                  <li className="flex gap-3">
                    <span className="font-bold min-w-[120px]">Cam Sensor:</span>
                    <span>Rear of Block (Top)</span>
                  </li>
                  <li className="flex gap-3">
                    <span className="font-bold min-w-[120px]">Knock Sensors:</span>
                    <span>In the Valley Cover (2 sensors)</span>
                  </li>
                  <li className="flex gap-3">
                    <span className="font-bold min-w-[120px]">Head Bolts:</span>
                    <span>All same length (early) or 2 lengths (late)</span>
                  </li>
                  <li className="flex gap-3">
                    <span className="font-bold min-w-[120px]">Throttle:</span>
                    <span>Drive-by-Cable (mostly) or DBW</span>
                  </li>
                </ul>
              </CardContent>
            </Card>

            <Card className="border-2 border-primary/20 bg-primary/5">
              <CardHeader className="bg-primary/10 border-b border-primary/20">
                <CardTitle className="font-mono text-lg flex justify-between items-center text-primary">
                  GENERATION 4
                  <Badge className="bg-primary text-primary-foreground">2005 - 2015+</Badge>
                </CardTitle>
              </CardHeader>
              <CardContent className="pt-6 space-y-4">
                <ul className="space-y-3 text-sm">
                  <li className="flex gap-3">
                    <span className="font-bold min-w-[120px]">Reluctor Wheel:</span>
                    <span>58x (Grey Crank Sensor)</span>
                  </li>
                  <li className="flex gap-3">
                    <span className="font-bold min-w-[120px]">Cam Sensor:</span>
                    <span>Front Timing Cover</span>
                  </li>
                  <li className="flex gap-3">
                    <span className="font-bold min-w-[120px]">Knock Sensors:</span>
                    <span>Side of Block (External)</span>
                  </li>
                  <li className="flex gap-3">
                    <span className="font-bold min-w-[120px]">Head Bolts:</span>
                    <span>All same length (M11)</span>
                  </li>
                  <li className="flex gap-3">
                    <span className="font-bold min-w-[120px]">Throttle:</span>
                    <span>Drive-by-Wire Only (4-bolt)</span>
                  </li>
                </ul>
              </CardContent>
            </Card>
          </div>
        </section>

        {/* The Head Port Myth */}
        <section className="space-y-6">
           <div className="flex items-center gap-2 mb-4">
            <div className="w-8 h-8 bg-primary text-primary-foreground flex items-center justify-center rounded-sm font-bold font-serif">2</div>
            <h2 className="text-2xl font-serif font-bold">The "Gen 4 = Rectangle Port" Myth</h2>
          </div>

          <Card className="border-l-4 border-l-yellow-500 bg-yellow-50/50 dark:bg-yellow-900/10">
            <CardContent className="pt-6">
              <div className="flex items-start gap-4">
                <AlertTriangle className="w-8 h-8 text-yellow-600 shrink-0" />
                <div className="space-y-4">
                  <h3 className="font-bold text-lg">Correction: Not all Gen 4 engines use Rectangle Port heads.</h3>
                  <p className="leading-relaxed">
                    A common misconception is that "Gen 4" automatically implies Rectangle Port (LS3/L92 style) heads. 
                    This is <strong>FALSE</strong>.
                  </p>
                  <p className="leading-relaxed">
                    GM produced millions of Gen 4 small blocks (specifically 4.8L and 5.3L truck engines like the LC9, LH6, and LMG) 
                    that feature <strong>Cathedral Port</strong> heads. These heads (often casting #799 or #243) are functionally identical 
                    to the legendary LS6 heads but come on standard Gen 4 truck engines.
                  </p>
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mt-4">
                    <div className="bg-background p-4 rounded border">
                      <div className="font-bold text-sm uppercase tracking-wider text-muted-foreground mb-1">Gen 4 with Rectangle Ports</div>
                      <ul className="list-disc list-inside text-sm font-mono">
                        <li>6.0L (LY6, L96)</li>
                        <li>6.2L (L92, LS3, L99, LSA, LS9)</li>
                        <li>7.0L (LS7)</li>
                      </ul>
                    </div>
                    <div className="bg-background p-4 rounded border">
                      <div className="font-bold text-sm uppercase tracking-wider text-muted-foreground mb-1">Gen 4 with Cathedral Ports</div>
                      <ul className="list-disc list-inside text-sm font-mono">
                        <li>4.8L (LY2, L20)</li>
                        <li>5.3L (LH6, LC9, LY5, LMG)</li>
                        <li>6.0L (LS2 - Early versions)</li>
                      </ul>
                    </div>
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>
        </section>

        {/* Interchangeability */}
        <section className="space-y-6">
          <div className="flex items-center gap-2 mb-4">
            <div className="w-8 h-8 bg-primary text-primary-foreground flex items-center justify-center rounded-sm font-bold font-serif">3</div>
            <h2 className="text-2xl font-serif font-bold">Interchangeability Matrix</h2>
          </div>

          <div className="border rounded-md overflow-hidden">
            <Table>
              <TableHeader className="bg-muted/50">
                <TableRow>
                  <TableHead className="font-bold text-foreground">Component</TableHead>
                  <TableHead>Gen 3 Block</TableHead>
                  <TableHead>Gen 4 Block</TableHead>
                  <TableHead>Notes</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                <TableRow>
                  <TableCell className="font-bold">Gen 3 Heads</TableCell>
                  <TableCell><CheckCircle2 className="w-5 h-5 text-green-600" /></TableCell>
                  <TableCell><CheckCircle2 className="w-5 h-5 text-green-600" /></TableCell>
                  <TableCell className="text-xs text-muted-foreground">Fits all. Check bore size for valve clearance.</TableCell>
                </TableRow>
                <TableRow>
                  <TableCell className="font-bold">Gen 4 Heads</TableCell>
                  <TableCell><CheckCircle2 className="w-5 h-5 text-green-600" /></TableCell>
                  <TableCell><CheckCircle2 className="w-5 h-5 text-green-600" /></TableCell>
                  <TableCell className="text-xs text-muted-foreground">
                    Fits all. <strong>Rectangle port heads require 4.000"+ bore</strong> (6.0L/6.2L only).
                    <br/>Will NOT fit 5.3L or 5.7L blocks due to valve shrouding.
                  </TableCell>
                </TableRow>
                <TableRow>
                  <TableCell className="font-bold">Front Covers</TableCell>
                  <TableCell><XCircle className="w-5 h-5 text-red-500" /></TableCell>
                  <TableCell><CheckCircle2 className="w-5 h-5 text-green-600" /></TableCell>
                  <TableCell className="text-xs text-muted-foreground">Gen 4 cover has cam sensor boss. Gen 3 does not.</TableCell>
                </TableRow>
                <TableRow>
                  <TableCell className="font-bold">Intake Manifolds</TableCell>
                  <TableCell className="text-yellow-600 font-bold text-xs">Port Dependent</TableCell>
                  <TableCell className="text-yellow-600 font-bold text-xs">Port Dependent</TableCell>
                  <TableCell className="text-xs text-muted-foreground">
                    Must match head port shape (Cathedral vs Rectangle). 
                    <br/>Bolt patterns differ (3-bolt vs 4-bolt throttle body).
                  </TableCell>
                </TableRow>
                <TableRow>
                  <TableCell className="font-bold">Valley Covers</TableCell>
                  <TableCell><XCircle className="w-5 h-5 text-red-500" /></TableCell>
                  <TableCell><XCircle className="w-5 h-5 text-red-500" /></TableCell>
                  <TableCell className="text-xs text-muted-foreground">
                    Gen 3 has knock sensor bosses. Gen 4 is flat (or has AFM solenoids). Not interchangeable.
                  </TableCell>
                </TableRow>
              </TableBody>
            </Table>
          </div>
        </section>

        {/* Camshaft Swap Guide */}
        <section className="space-y-6">
          <div className="flex items-center gap-2 mb-4">
            <div className="w-8 h-8 bg-primary text-primary-foreground flex items-center justify-center rounded-sm font-bold font-serif">4</div>
            <h2 className="text-2xl font-serif font-bold">Camshaft Swap Logic (Gen 4 Cam in Gen 3 Block)</h2>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            <Card className="border-l-4 border-l-red-500 bg-red-50/50 dark:bg-red-900/10">
              <CardHeader className="pb-2">
                <CardTitle className="text-red-700 dark:text-red-400 flex items-center gap-2 text-lg">
                  <XCircle className="w-5 h-5" />
                  AVOID: 1-Bolt Cams (Stock LS3/L92)
                </CardTitle>
              </CardHeader>
              <CardContent className="text-sm space-y-3">
                <p>
                  Trying to run a stock <strong>1-Bolt Gen 4 Cam</strong> (like an LS3 cam) in a Gen 3 block is generally <strong>not worth the effort</strong>.
                </p>
                <ul className="list-disc list-inside space-y-1 text-muted-foreground">
                  <li>Requires a specific 1-bolt 24x timing gear (hard to find/expensive).</li>
                  <li>Requires converting to a front timing cover with cam sensor.</li>
                  <li>Requires extending the cam sensor harness.</li>
                </ul>
                <p className="font-semibold text-red-600 dark:text-red-400">
                  Verdict: Just buy a Gen 3 specific cam. It saves money and headache.
                </p>
              </CardContent>
            </Card>

            <Card className="border-l-4 border-l-green-500 bg-green-50/50 dark:bg-green-900/10">
              <CardHeader className="pb-2">
                <CardTitle className="text-green-700 dark:text-green-400 flex items-center gap-2 text-lg">
                  <CheckCircle2 className="w-5 h-5" />
                  DO IT: 3-Bolt Cams (LS9 / Aftermarket)
                </CardTitle>
              </CardHeader>
              <CardContent className="text-sm space-y-3">
                <p>
                  Running a <strong>3-Bolt Gen 4 Cam</strong> (like the LS9 or aftermarket) in a Gen 3 block is a viable and popular upgrade.
                </p>
                <div className="bg-background/80 p-3 rounded border border-green-200 dark:border-green-800">
                  <span className="font-bold text-xs uppercase block mb-1">Required Changes:</span>
                  <ul className="list-disc list-inside space-y-1 text-muted-foreground text-xs">
                    <li><strong>Timing Cover:</strong> Must swap to Gen 4 front cover (with cam sensor boss).</li>
                    <li><strong>Cam Gear:</strong> Use a 4x (Gen 4) or 1x (Gen 3) 3-bolt gear depending on your ECU (24x vs 58x).</li>
                    <li><strong>Sensor Extension:</strong> Extension harness to reach front cover.</li>
                  </ul>
                </div>
                <p className="font-semibold text-green-600 dark:text-green-400 mt-2">
                  Verdict: Worth it for the LS9 cam's value or if you already have the parts.
                </p>
              </CardContent>
            </Card>
          </div>
        </section>

        {/* Pushrod Differences */}
        <section className="space-y-6">
          <div className="flex items-center gap-2 mb-4">
            <div className="w-8 h-8 bg-primary text-primary-foreground flex items-center justify-center rounded-sm font-bold font-serif">5</div>
            <h2 className="text-2xl font-serif font-bold">Pushrod Lengths: Standard vs. LS7/LS9</h2>
          </div>

          <Card className="border-l-4 border-l-blue-500 bg-blue-50/50 dark:bg-blue-900/10">
            <CardHeader>
              <CardTitle className="flex items-center gap-2 text-lg">
                <Info className="w-5 h-5 text-blue-600" />
                Critical Difference: 7.400" vs 7.800"
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <p className="text-sm leading-relaxed">
                While most LS engines share a common pushrod length, the LS7 (7.0L) and LS9 (6.2L SC) are significant outliers due to their unique cylinder head architecture.
              </p>
              
              <div className="grid grid-cols-1 md:grid-cols-2 gap-6 mt-2">
                <div className="bg-background p-4 rounded border border-border/50">
                  <div className="flex justify-between items-center mb-2 border-b border-border/50 pb-2">
                    <span className="font-bold text-sm uppercase tracking-wider">Standard LS Pushrod</span>
                    <Badge variant="outline" className="font-mono">7.400"</Badge>
                  </div>
                  <ul className="list-disc list-inside text-sm space-y-1 text-muted-foreground">
                    <li><strong>Applications:</strong> LS1, LS6, LS2, LS3, L99, LSA, and almost all Truck engines (4.8/5.3/6.0/6.2).</li>
                    <li><strong>Reason:</strong> Standard deck height and standard 15° or 12° valve angle head geometry.</li>
                    <li><strong>Note:</strong> Always measure when changing cams, but 7.400" is the baseline.</li>
                  </ul>
                </div>

                <div className="bg-background p-4 rounded border border-border/50">
                  <div className="flex justify-between items-center mb-2 border-b border-border/50 pb-2">
                    <span className="font-bold text-sm uppercase tracking-wider">LS7 / LS9 Pushrod</span>
                    <Badge variant="default" className="font-mono bg-blue-600">7.800"</Badge>
                  </div>
                  <ul className="list-disc list-inside text-sm space-y-1 text-muted-foreground">
                    <li><strong>Applications:</strong> LS7 (Z06) and LS9 (ZR1) ONLY.</li>
                    <li><strong>Reason:</strong> These heads feature a raised rocker arm pedestal and 12° valve angle geometry that sits higher up. The taller deck on the head requires a significantly longer pushrod.</li>
                    <li><strong>Warning:</strong> Using a 7.400" pushrod in an LS7/LS9 engine will result in zero valve lift and massive lash.</li>
                  </ul>
                </div>
              </div>
            </CardContent>
          </Card>
        </section>

        {/* Throttle Systems: DBW vs DBC */}
        <section className="space-y-6">
          <div className="flex items-center gap-2 mb-4">
            <div className="w-8 h-8 bg-primary text-primary-foreground flex items-center justify-center rounded-sm font-bold font-serif">6</div>
            <h2 className="text-2xl font-serif font-bold">Throttle Systems: Cable vs. Wire</h2>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
            <Card className="border-t-4 border-t-blue-500 shadow-md">
              <CardHeader>
                <CardTitle className="text-lg">Drive-by-Cable (DBC)</CardTitle>
                <Badge variant="secondary" className="w-fit mt-1">Gen 3 Only</Badge>
              </CardHeader>
              <CardContent className="text-sm space-y-3">
                <p className="text-muted-foreground">
                  The traditional mechanical setup. A physical steel cable connects the pedal directly to the throttle body blade.
                </p>
                <ul className="list-disc list-inside space-y-1 text-xs font-medium">
                  <li>Found on: 1997-2002 Camaro/Firebird, older Trucks/Vans.</li>
                  <li>Mount: 3-Bolt.</li>
                  <li>IAC Valve: Separate idle air control motor.</li>
                  <li>TPS: Separate throttle position sensor.</li>
                </ul>
                <div className="bg-blue-50 dark:bg-blue-900/10 p-2 rounded text-xs text-blue-700 dark:text-blue-300 font-bold">
                  Pro: Easiest for swaps (no electronic pedal needed).
                </div>
              </CardContent>
            </Card>

            <Card className="border-t-4 border-t-orange-500 shadow-md">
              <CardHeader>
                <CardTitle className="text-lg">Gen 3 Drive-by-Wire</CardTitle>
                <Badge variant="secondary" className="w-fit mt-1">Early Electronic</Badge>
              </CardHeader>
              <CardContent className="text-sm space-y-3">
                <p className="text-muted-foreground">
                  Early electronic throttle. Requires a separate TAC (Throttle Actuator Control) module (black box) mounted near the ECU.
                </p>
                <ul className="list-disc list-inside space-y-1 text-xs font-medium">
                  <li>Found on: 1997-2004 Corvette, 2003-2007 Trucks/SUVs.</li>
                  <li>Mount: 3-Bolt.</li>
                  <li>Complexity: High. Pedal, TAC module, and Throttle Body must match specifically.</li>
                </ul>
                <div className="bg-orange-50 dark:bg-orange-900/10 p-2 rounded text-xs text-orange-700 dark:text-orange-300 font-bold">
                  Warning: Mismatched TAC modules cause "Reduced Engine Power" mode.
                </div>
              </CardContent>
            </Card>

            <Card className="border-t-4 border-t-purple-500 shadow-md">
              <CardHeader>
                <CardTitle className="text-lg">Gen 4 Drive-by-Wire</CardTitle>
                <Badge variant="secondary" className="w-fit mt-1">Modern Standard</Badge>
              </CardHeader>
              <CardContent className="text-sm space-y-3">
                <p className="text-muted-foreground">
                  Modern simplified system. The TAC module is integrated into the ECU (E38/E67), eliminating the separate black box.
                </p>
                <ul className="list-disc list-inside space-y-1 text-xs font-medium">
                  <li>Found on: 2005+ Corvette, 2007+ Trucks, Gen 5 LT.</li>
                  <li>Mount: 4-Bolt.</li>
                  <li>Blade: Silver (LS2/truck) or Gold (LS3/LS7). Not interchangeable!</li>
                </ul>
                <div className="bg-purple-50 dark:bg-purple-900/10 p-2 rounded text-xs text-purple-700 dark:text-purple-300 font-bold">
                  Note: Gold blade throttles open opposite direction of Silver. Requires tuning.
                </div>
              </CardContent>
            </Card>
          </div>
        </section>

        {/* Injector Adapters */}
        <section className="space-y-6">
          <div className="flex items-center gap-2 mb-4">
            <div className="w-8 h-8 bg-primary text-primary-foreground flex items-center justify-center rounded-sm font-bold font-serif">7</div>
            <h2 className="text-2xl font-serif font-bold">The "Cross-Over" List (Adapters)</h2>
          </div>

          <Card className="border-2 border-border/60 bg-card/50">
            <CardHeader className="bg-muted/20 border-b border-border/20">
              <CardTitle className="text-lg">Injector Connector Map</CardTitle>
            </CardHeader>
            <CardContent className="pt-6 space-y-6">
              <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                <div className="p-4 rounded border bg-background">
                  <div className="font-bold text-sm uppercase tracking-wider mb-2">Multec 2</div>
                  <p className="text-xs text-muted-foreground mb-2">The weird "Mini-Delphi" connector.</p>
                  <Badge variant="secondary">Gen 3 Trucks (LM7/LQ4)</Badge>
                </div>
                <div className="p-4 rounded border bg-background">
                  <div className="font-bold text-sm uppercase tracking-wider mb-2">EV1</div>
                  <p className="text-xs text-muted-foreground mb-2">Standard rectangular connector.</p>
                  <Badge variant="secondary">Gen 3 Cars (LS1)</Badge>
                </div>
                <div className="p-4 rounded border bg-background">
                  <div className="font-bold text-sm uppercase tracking-wider mb-2">EV6 / USCAR</div>
                  <p className="text-xs text-muted-foreground mb-2">Small square connector.</p>
                  <Badge variant="default">All Gen 4 Engines</Badge>
                </div>
              </div>

              <div className="bg-yellow-50/50 dark:bg-yellow-900/10 p-4 rounded border-l-4 border-l-yellow-500">
                <h4 className="font-bold text-yellow-700 dark:text-yellow-400 mb-2 flex items-center gap-2">
                  <AlertTriangle className="w-4 h-4" />
                  Scenario: L96 (Flex Fuel) Injectors on Gen 3 Truck Harness
                </h4>
                <p className="text-sm text-muted-foreground mb-3">
                  If you are upgrading a Gen 3 truck with superior Gen 4 injectors, you cannot just plug them in. You need two specific items:
                </p>
                <ul className="list-disc list-inside text-sm font-medium space-y-2">
                  <li>
                    <span className="font-bold">Multec-to-EV6 Adapters:</span> To convert the harness plug to fit the injector.
                  </li>
                  <li>
                    <span className="font-bold">Top Spacers / Injector Hats:</span> Gen 4 injectors are shorter. You must space the fuel rail down OR use "Injector Hats" to extend the injector height to reach the Gen 3 rail.
                  </li>
                </ul>
              </div>
            </CardContent>
          </Card>
        </section>

        {/* Oil Pumps */}
        <section className="space-y-6">
          <div className="flex items-center gap-2 mb-4">
            <div className="w-8 h-8 bg-primary text-primary-foreground flex items-center justify-center rounded-sm font-bold font-serif">8</div>
            <h2 className="text-2xl font-serif font-bold">Oil Pumps: Standard vs. High Volume</h2>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            <Card className="border-t-4 border-t-zinc-500 shadow-md">
              <CardHeader>
                <CardTitle className="text-lg">Standard Volume (Gen 3)</CardTitle>
                <Badge variant="secondary" className="w-fit mt-1">GM 12586665 / Melling M295</Badge>
              </CardHeader>
              <CardContent className="text-sm space-y-3">
                <p className="text-muted-foreground">
                  The base pump found on almost all Gen 3 engines and Gen 4 engines <strong>without</strong> AFM or VVT.
                </p>
                <ul className="list-disc list-inside space-y-1 text-xs font-medium">
                  <li><strong>Flow:</strong> ~6.8 GPM</li>
                  <li><strong>Best For:</strong> LS1, LS6, LQ4, LQ9, LS2, and "Delete" builds.</li>
                </ul>
                <div className="bg-zinc-100 dark:bg-zinc-800 p-2 rounded text-xs font-bold">
                  Note: Do NOT use on engines with active AFM/DOD.
                </div>
              </CardContent>
            </Card>

            <Card className="border-t-4 border-t-blue-600 shadow-md">
              <CardHeader>
                <CardTitle className="text-lg">High Volume (Gen 4)</CardTitle>
                <Badge className="w-fit mt-1 bg-blue-600">GM 12612289 / Melling M365</Badge>
              </CardHeader>
              <CardContent className="text-sm space-y-3">
                <p className="text-muted-foreground">
                  Deeper gears move ~30% more oil. Required for Gen 4 engines with complex valvetrains that use oil pressure to operate.
                </p>
                <ul className="list-disc list-inside space-y-1 text-xs font-medium">
                  <li><strong>Flow:</strong> ~8.8 - 9.5 GPM</li>
                  <li><strong>Best For:</strong> L99, L92, LSA, and any 5.3L/6.0L with AFM/VVT.</li>
                </ul>
                <div className="bg-blue-50 dark:bg-blue-900/10 p-2 rounded text-xs text-blue-700 dark:text-blue-300 font-bold">
                  Rule: Mandatory for VVT/AFM. Optional insurance for others.
                </div>
              </CardContent>
            </Card>
          </div>

          <Card className="border border-border/60 bg-muted/20">
            <CardContent className="pt-6 flex items-start gap-4">
               <Info className="w-6 h-6 text-primary mt-1 shrink-0" />
               <div className="space-y-2">
                 <h3 className="font-bold text-base">Interchangeability Rule</h3>
                 <p className="text-sm text-muted-foreground">
                   You <strong>CAN</strong> install a Gen 4 High Volume pump on a Gen 3 engine (it fits perfectly and provides extra oil pressure insurance).
                 </p>
                 <p className="text-sm text-muted-foreground">
                   You <strong>CANNOT</strong> install a Gen 3 Standard pump on a Gen 4 AFM/VVT engine (it will starve the lifters/phasers and cause failure).
                 </p>
               </div>
            </CardContent>
          </Card>
        </section>

        {/* Oil Pickup O-Rings */}
        <section className="space-y-6">
          <div className="flex items-center gap-2 mb-4">
            <div className="w-8 h-8 bg-primary text-primary-foreground flex items-center justify-center rounded-sm font-bold font-serif">9</div>
            <h2 className="text-2xl font-serif font-bold">The O-Ring Trap: Blue vs. Red/Green</h2>
          </div>

          <Card className="border-l-4 border-l-red-500 bg-red-50/50 dark:bg-red-900/10 mb-6">
             <CardContent className="pt-6 flex items-start gap-4">
               <AlertTriangle className="w-8 h-8 text-red-600 shrink-0" />
               <div className="space-y-2">
                 <h3 className="font-bold text-lg text-red-700 dark:text-red-400">Why this matters</h3>
                 <p className="text-sm text-muted-foreground">
                   Using the wrong O-ring is the #1 cause of low oil pressure after a cam swap or rebuild. The pump will suck air instead of oil, aerating the fluid and destroying bearings.
                 </p>
                 <p className="font-bold text-sm">
                   Identify your pickup tube style visually before installing the O-ring. Do not guess based on year alone.
                 </p>
               </div>
            </CardContent>
          </Card>

          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            <Card className="border-t-4 border-t-blue-500 shadow-md">
              <CardHeader>
                <CardTitle className="text-lg flex justify-between">
                  <span>The "Blue" Style</span>
                  <Badge className="bg-blue-500 hover:bg-blue-600">Thick / Large</Badge>
                </CardTitle>
                <p className="text-xs text-muted-foreground font-mono mt-1">GM 12557752</p>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="bg-muted p-3 rounded text-sm font-medium border">
                  <span className="block text-xs uppercase text-muted-foreground mb-1">Visual ID (Tube End)</span>
                  Smooth / Straight Pipe. No welded washer or indentation.
                </div>
                
                <ul className="space-y-2 text-sm">
                  <li className="flex justify-between border-b pb-1">
                    <span className="text-muted-foreground">Color</span>
                    <span className="font-medium">Blue (Factory) / Black (Melling)</span>
                  </li>
                  <li className="flex justify-between border-b pb-1">
                    <span className="text-muted-foreground">Inner Diameter</span>
                    <span className="font-medium">~23.60mm (Larger)</span>
                  </li>
                  <li className="flex justify-between border-b pb-1">
                    <span className="text-muted-foreground">Tube Design</span>
                    <span className="font-medium">1st Design (Early)</span>
                  </li>
                </ul>
                <div className="text-xs text-muted-foreground">
                  <strong>Common In:</strong> Early Gen 3 (LS1, LM7).
                </div>
              </CardContent>
            </Card>

            <Card className="border-t-4 border-t-green-600 shadow-md">
              <CardHeader>
                <CardTitle className="text-lg flex justify-between">
                  <span>The "Red" Style</span>
                  <Badge className="bg-green-600 hover:bg-green-700">Thin / Tapered</Badge>
                </CardTitle>
                <p className="text-xs text-muted-foreground font-mono mt-1">GM 12584922</p>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="bg-muted p-3 rounded text-sm font-medium border">
                  <span className="block text-xs uppercase text-muted-foreground mb-1">Visual ID (Tube End)</span>
                  Grooved / Tapered. Has a rolled groove or step near the end.
                </div>

                <ul className="space-y-2 text-sm">
                  <li className="flex justify-between border-b pb-1">
                    <span className="text-muted-foreground">Color</span>
                    <span className="font-medium">Red/Orange (Factory) / Green (Melling)</span>
                  </li>
                  <li className="flex justify-between border-b pb-1">
                    <span className="text-muted-foreground">Inner Diameter</span>
                    <span className="font-medium">~20.69mm (Smaller)</span>
                  </li>
                  <li className="flex justify-between border-b pb-1">
                    <span className="text-muted-foreground">Tube Design</span>
                    <span className="font-medium">2nd Design (Late)</span>
                  </li>
                </ul>
                <div className="text-xs text-muted-foreground">
                  <strong>Common In:</strong> Gen 4 (LS2, LS3, LY6) & Late Gen 3 (LQ4/LQ9).
                </div>
              </CardContent>
            </Card>
          </div>
        </section>

        {/* Transmission & Flexplate Guide */}
        <section className="space-y-6">
          <div className="flex items-center gap-2 mb-4">
            <div className="w-8 h-8 bg-primary text-primary-foreground flex items-center justify-center rounded-sm font-bold font-serif">10</div>
            <h2 className="text-2xl font-serif font-bold">Transmission & Flexplate Guide</h2>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            <Card className="border-l-4 border-l-orange-500 bg-orange-50/50 dark:bg-orange-900/10">
              <CardHeader className="pb-2">
                <CardTitle className="text-orange-700 dark:text-orange-400 flex items-center gap-2 text-lg">
                  <AlertTriangle className="w-5 h-5" />
                  Critical Warnings
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-4 pt-2">
                <div className="space-y-2">
                  <div className="font-bold text-sm">1. Is your engine a 1999-2000 6.0L?</div>
                  <p className="text-xs text-muted-foreground bg-background p-2 rounded border">
                    <span className="font-bold text-red-600">WARNING: Long Crank Flange.</span> These early iron heads (often LQ4) have a crankshaft flange that sticks out 0.400" further. They will NOT bolt to standard LS transmissions without specific torque converters or flywheel setups.
                  </p>
                </div>
                <div className="space-y-2">
                  <div className="font-bold text-sm">2. Is it LSA, LS9, or LSX?</div>
                  <p className="text-xs text-muted-foreground bg-background p-2 rounded border">
                    <span className="font-bold text-red-600">WARNING: 8-Bolt Flange.</span> Standard LS cranks have 6 bolts. High-performance factory engines use 8 bolts. You MUST buy an 8-bolt specific flexplate/flywheel.
                  </p>
                </div>
              </CardContent>
            </Card>

            <Card className="border-2 border-border/60">
              <CardHeader className="bg-muted/20 border-b border-border/20 pb-4">
                <CardTitle className="text-lg">Flexplate Selection Rule</CardTitle>
              </CardHeader>
              <CardContent className="pt-4 space-y-4">
                <div className="flex items-center justify-between border-b pb-2">
                  <div>
                    <div className="font-bold text-sm">4L60E / 6L80 / 6L90</div>
                    <div className="text-xs text-muted-foreground">Standard automatic builds</div>
                  </div>
                  <Badge variant="outline" className="ml-2 whitespace-nowrap">Dished Flexplate</Badge>
                </div>
                
                <div className="flex items-center justify-between border-b pb-2">
                  <div>
                    <div className="font-bold text-sm">4L80E / TH400 / TH350</div>
                    <div className="text-xs text-muted-foreground">Heavy duty / Older trans</div>
                  </div>
                  <div className="text-right">
                    <Badge variant="default" className="whitespace-nowrap mb-1">Flat Flexplate</Badge>
                    <div className="text-[10px] font-bold text-primary">+ Crank Spacer</div>
                  </div>
                </div>

                <div className="flex items-center justify-between">
                  <div>
                    <div className="font-bold text-sm">Manual (T56 / TR6060)</div>
                    <div className="text-xs text-muted-foreground">Stick shift swaps</div>
                  </div>
                  <div className="text-right">
                    <Badge variant="secondary" className="whitespace-nowrap mb-1">Specific Flywheel</Badge>
                    <div className="text-[10px] font-bold text-muted-foreground">Watch Pilot Bearing!</div>
                  </div>
                </div>
              </CardContent>
            </Card>
          </div>
        </section>

        {/* The 8th Digit Trick */}
        <section className="space-y-6">
          <div className="flex items-center gap-2 mb-4">
            <div className="w-8 h-8 bg-primary text-primary-foreground flex items-center justify-center rounded-sm font-bold font-serif">11</div>
            <h2 className="text-2xl font-serif font-bold">The "8th Digit Trick"</h2>
          </div>

          <Card className="border-l-4 border-l-green-500 bg-green-50/50 dark:bg-green-900/10 mb-6">
            <CardContent className="pt-6 flex items-start gap-4">
              <CheckCircle2 className="w-8 h-8 text-green-600 shrink-0" />
              <div className="space-y-2">
                <h3 className="font-bold text-lg text-green-700 dark:text-green-400">Quick Engine ID from VIN</h3>
                <p className="text-sm text-muted-foreground">
                  You can instantly identify the engine by looking at the <strong>8th character of the VIN</strong>. 
                  This trick works for GM trucks (1999-2013) and cars (1997-2013, with Caprice PPV/SS Sedan extending to 2017).
                </p>
                <p className="text-xs font-medium">
                  Example VIN: 1GCEK19<strong className="text-green-600 text-base">T</strong>X1E123456 → The "T" means it's an LM7 5.3L Iron Block.
                </p>
              </div>
            </CardContent>
          </Card>

          {/* Trucks Table */}
          <h3 className="text-lg font-bold mt-6 mb-2 flex items-center gap-2">
            <Badge variant="secondary" className="font-mono">TRUCKS & SUVs</Badge>
            <span className="text-sm text-muted-foreground font-normal">1999-2013 (Silverado, Sierra, Tahoe, Yukon, Suburban, Avalanche)</span>
          </h3>
          <div className="border rounded-md overflow-hidden">
            <Table>
              <TableHeader className="bg-muted/50">
                <TableRow>
                  <TableHead className="font-bold text-foreground w-20">8th Digit</TableHead>
                  <TableHead className="font-bold text-foreground">Engine Code</TableHead>
                  <TableHead>Size</TableHead>
                  <TableHead>Block</TableHead>
                  <TableHead>Notes</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                <TableRow>
                  <TableCell className="font-mono text-lg font-bold text-primary">V</TableCell>
                  <TableCell className="font-bold">LR4</TableCell>
                  <TableCell>4.8L</TableCell>
                  <TableCell><Badge variant="secondary">Iron</Badge></TableCell>
                  <TableCell className="text-sm text-muted-foreground">The smallest Vortec. 24x reluctor.</TableCell>
                </TableRow>
                <TableRow>
                  <TableCell className="font-mono text-lg font-bold text-primary">T</TableCell>
                  <TableCell className="font-bold">LM7</TableCell>
                  <TableCell>5.3L</TableCell>
                  <TableCell><Badge variant="secondary">Iron</Badge></TableCell>
                  <TableCell className="text-sm text-muted-foreground">The standard "Vortec 5300". 24x.</TableCell>
                </TableRow>
                <TableRow>
                  <TableCell className="font-mono text-lg font-bold text-primary">Z</TableCell>
                  <TableCell className="font-bold">L59</TableCell>
                  <TableCell>5.3L</TableCell>
                  <TableCell><Badge variant="secondary">Iron</Badge></TableCell>
                  <TableCell className="text-sm text-muted-foreground">Flex Fuel version of the LM7.</TableCell>
                </TableRow>
                <TableRow>
                  <TableCell className="font-mono text-lg font-bold text-primary">U</TableCell>
                  <TableCell className="font-bold">LQ4</TableCell>
                  <TableCell>6.0L</TableCell>
                  <TableCell><Badge variant="secondary">Iron</Badge></TableCell>
                  <TableCell className="text-sm text-muted-foreground">Low compression 6.0L. (See note below).</TableCell>
                </TableRow>
                <TableRow>
                  <TableCell className="font-mono text-lg font-bold text-primary">N</TableCell>
                  <TableCell className="font-bold">LQ9</TableCell>
                  <TableCell>6.0L</TableCell>
                  <TableCell><Badge variant="secondary">Iron</Badge></TableCell>
                  <TableCell className="text-sm text-muted-foreground">"High Output" 6.0L (Escalade/SS). Flat top pistons.</TableCell>
                </TableRow>
                <TableRow className="bg-blue-50/50 dark:bg-blue-900/10">
                  <TableCell className="font-mono text-lg font-bold text-blue-600">B</TableCell>
                  <TableCell className="font-bold text-blue-700 dark:text-blue-400">L33</TableCell>
                  <TableCell>5.3L</TableCell>
                  <TableCell><Badge className="bg-blue-600">Alum</Badge></TableCell>
                  <TableCell className="text-sm text-muted-foreground">"Vortec HO". High compression, reliable.</TableCell>
                </TableRow>
                <TableRow>
                  <TableCell className="font-mono text-lg font-bold text-primary">M</TableCell>
                  <TableCell className="font-bold">LH6</TableCell>
                  <TableCell>5.3L</TableCell>
                  <TableCell><Badge className="bg-blue-600">Alum</Badge></TableCell>
                  <TableCell className="text-sm text-muted-foreground">Gen 4 with AFM (Active Fuel Management).</TableCell>
                </TableRow>
                <TableRow>
                  <TableCell className="font-mono text-lg font-bold text-primary">0</TableCell>
                  <TableCell className="font-bold">LMG</TableCell>
                  <TableCell>5.3L</TableCell>
                  <TableCell><Badge variant="secondary">Iron</Badge></TableCell>
                  <TableCell className="text-sm text-muted-foreground">Gen 4 Flex Fuel. Very common in 2007+ trucks.</TableCell>
                </TableRow>
                <TableRow>
                  <TableCell className="font-mono text-lg font-bold text-primary">J</TableCell>
                  <TableCell className="font-bold">LY5</TableCell>
                  <TableCell>5.3L</TableCell>
                  <TableCell><Badge variant="secondary">Iron</Badge></TableCell>
                  <TableCell className="text-sm text-muted-foreground">Gen 4 Iron block (Non-Flex).</TableCell>
                </TableRow>
                <TableRow>
                  <TableCell className="font-mono text-lg font-bold text-primary">K</TableCell>
                  <TableCell className="font-bold">LY6</TableCell>
                  <TableCell>6.0L</TableCell>
                  <TableCell><Badge variant="secondary">Iron</Badge></TableCell>
                  <TableCell className="text-sm text-muted-foreground">Gen 4 replacement for the LQ4. Rectangle ports.</TableCell>
                </TableRow>
                <TableRow className="bg-green-50/50 dark:bg-green-900/10">
                  <TableCell className="font-mono text-lg font-bold text-green-600">8</TableCell>
                  <TableCell className="font-bold text-green-700 dark:text-green-400">L92</TableCell>
                  <TableCell>6.2L</TableCell>
                  <TableCell><Badge className="bg-green-600">Alum</Badge></TableCell>
                  <TableCell className="text-sm text-muted-foreground">The "Truck LS3". High value!</TableCell>
                </TableRow>
              </TableBody>
            </Table>
          </div>

          <Card className="border border-border/60 bg-muted/20">
            <CardContent className="pt-6 flex items-start gap-4">
              <Info className="w-6 h-6 text-primary mt-1 shrink-0" />
              <div className="space-y-2">
                <h3 className="font-bold text-base">Truck Notes</h3>
                <ul className="space-y-2 text-sm text-muted-foreground">
                  <li>• <strong>LQ4 vs LQ9:</strong> Both are 6.0L iron blocks, but the LQ9 (digit "N") has flat-top pistons for higher compression. Found in Escalade and SS trucks.</li>
                  <li>• <strong>L33 "Vortec HO":</strong> The hidden gem. Aluminum 5.3L with high compression. Often found in 2005-2007 Silverado SS and TrailBlazer SS.</li>
                  <li>• <strong>L92 "Truck LS3":</strong> Rectangle port aluminum 6.2L. Basically an LS3 in truck form. Highly desirable for swaps.</li>
                  <li>• <strong>Digit "0" (Zero):</strong> Don't confuse with "O" (letter). The LMG uses the numeral zero.</li>
                </ul>
              </div>
            </CardContent>
          </Card>

          {/* Cars Table */}
          <h3 className="text-lg font-bold mt-8 mb-2 flex items-center gap-2">
            <Badge className="font-mono bg-primary">CARS</Badge>
            <span className="text-sm text-muted-foreground font-normal">1997-2013 (Corvette, Camaro, GTO, CTS-V, G8, SS Sedan, Caprice PPV to 2017)</span>
          </h3>
          <div className="border rounded-md overflow-hidden">
            <Table>
              <TableHeader className="bg-muted/50">
                <TableRow>
                  <TableHead className="font-bold text-foreground w-20">8th Digit</TableHead>
                  <TableHead className="font-bold text-foreground">Engine Code</TableHead>
                  <TableHead>Size</TableHead>
                  <TableHead>Block</TableHead>
                  <TableHead>Notes</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                <TableRow className="bg-yellow-50/50 dark:bg-yellow-900/10">
                  <TableCell className="font-mono text-lg font-bold text-yellow-600">G</TableCell>
                  <TableCell className="font-bold text-yellow-700 dark:text-yellow-400">LS1</TableCell>
                  <TableCell>5.7L</TableCell>
                  <TableCell><Badge className="bg-blue-600">Alum</Badge></TableCell>
                  <TableCell className="text-sm text-muted-foreground">The original legend.</TableCell>
                </TableRow>
                <TableRow>
                  <TableCell className="font-mono text-lg font-bold text-primary">S</TableCell>
                  <TableCell className="font-bold">LS6</TableCell>
                  <TableCell>5.7L</TableCell>
                  <TableCell><Badge className="bg-blue-600">Alum</Badge></TableCell>
                  <TableCell className="text-sm text-muted-foreground">Z06 Corvette / CTS-V.</TableCell>
                </TableRow>
                <TableRow className="bg-orange-50/50 dark:bg-orange-900/10">
                  <TableCell className="font-mono text-lg font-bold text-orange-600">U</TableCell>
                  <TableCell className="font-bold text-orange-700 dark:text-orange-400">LS2</TableCell>
                  <TableCell>6.0L</TableCell>
                  <TableCell><Badge className="bg-blue-600">Alum</Badge></TableCell>
                  <TableCell className="text-sm text-muted-foreground">Gen 4. (Watch out: "U" is also a truck LQ4!)</TableCell>
                </TableRow>
                <TableRow>
                  <TableCell className="font-mono text-lg font-bold text-primary">W</TableCell>
                  <TableCell className="font-bold">LS3</TableCell>
                  <TableCell>6.2L</TableCell>
                  <TableCell><Badge className="bg-blue-600">Alum</Badge></TableCell>
                  <TableCell className="text-sm text-muted-foreground">The standard for Gen 4 power.</TableCell>
                </TableRow>
                <TableRow className="bg-red-50/50 dark:bg-red-900/10">
                  <TableCell className="font-mono text-lg font-bold text-red-600">E</TableCell>
                  <TableCell className="font-bold text-red-700 dark:text-red-400">LS7</TableCell>
                  <TableCell>7.0L</TableCell>
                  <TableCell><Badge className="bg-blue-600">Alum</Badge></TableCell>
                  <TableCell className="text-sm text-muted-foreground">427ci Z06 engine. Dry sump.</TableCell>
                </TableRow>
                <TableRow>
                  <TableCell className="font-mono text-lg font-bold text-primary">J</TableCell>
                  <TableCell className="font-bold">L99</TableCell>
                  <TableCell>6.2L</TableCell>
                  <TableCell><Badge className="bg-blue-600">Alum</Badge></TableCell>
                  <TableCell className="text-sm text-muted-foreground">Auto trans version of LS3 (has VVT/AFM).</TableCell>
                </TableRow>
              </TableBody>
            </Table>
          </div>

          <Card className="border border-border/60 bg-muted/20">
            <CardContent className="pt-6 flex items-start gap-4">
              <AlertTriangle className="w-6 h-6 text-orange-500 mt-1 shrink-0" />
              <div className="space-y-2">
                <h3 className="font-bold text-base">Car Notes & Warnings</h3>
                <ul className="space-y-2 text-sm text-muted-foreground">
                  <li>• <strong>LS1 "G":</strong> The engine that started the revolution. Found in 1997-2004 Corvette and 1998-2002 F-Body (Camaro/Firebird).</li>
                  <li>• <strong>LS2 "U" Conflict:</strong> The letter "U" is used for BOTH the LS2 (cars) and LQ4 (trucks). Check if it's a car or truck VIN to determine which engine it is!</li>
                  <li>• <strong>LS7 "E":</strong> The mighty 7.0L. Only found in the 2006-2013 Z06 Corvette. Features titanium rods and dry sump oiling.</li>
                  <li>• <strong>L99 vs LS3:</strong> Both are 6.2L, but L99 (digit "J") has AFM/VVT for the automatic Camaros. Less aggressive cam than manual LS3.</li>
                </ul>
              </div>
            </CardContent>
          </Card>
        </section>

        {/* Oil Pan Selection Guide */}
        <section className="space-y-6">
          <div className="flex items-center gap-2 mb-4">
            <div className="w-8 h-8 bg-primary text-primary-foreground flex items-center justify-center rounded-sm font-bold font-serif">12</div>
            <h2 className="text-2xl font-serif font-bold">Oil Pan Selection Guide</h2>
          </div>

          <Card className="border-l-4 border-l-blue-500 bg-blue-50/50 dark:bg-blue-900/10 mb-6">
            <CardContent className="pt-6 flex items-start gap-4">
              <Info className="w-8 h-8 text-blue-600 shrink-0" />
              <div className="space-y-2">
                <h3 className="font-bold text-lg text-blue-700 dark:text-blue-400">Why Oil Pan Selection Matters</h3>
                <p className="text-sm text-muted-foreground">
                  The oil pan is often the most critical fitment piece in an LS swap. Different pans have different sump locations (front vs rear), 
                  depths, and profiles that determine steering rack clearance, ground clearance, and crossmember compatibility.
                </p>
              </div>
            </CardContent>
          </Card>

          <div className="border rounded-md overflow-hidden">
            <Table>
              <TableHeader className="bg-muted/50">
                <TableRow>
                  <TableHead className="font-bold text-foreground">Pan Application</TableHead>
                  <TableHead>Sump Location</TableHead>
                  <TableHead>Capacity</TableHead>
                  <TableHead>Best Use Case</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                <TableRow className="bg-green-50/50 dark:bg-green-900/10">
                  <TableCell className="font-bold text-green-700 dark:text-green-400">F-Body (Camaro 98-02)</TableCell>
                  <TableCell>Rear</TableCell>
                  <TableCell>5.5 qt</TableCell>
                  <TableCell className="text-sm text-muted-foreground">Muscle car swaps (low profile) - The "Gold Standard"</TableCell>
                </TableRow>
                <TableRow>
                  <TableCell className="font-bold">Hummer H3 (LH8)</TableCell>
                  <TableCell>Rear</TableCell>
                  <TableCell>5.5 qt</TableCell>
                  <TableCell className="text-sm text-muted-foreground">Swaps where ground clearance isn't an issue</TableCell>
                </TableRow>
                <TableRow>
                  <TableCell className="font-bold">CTS-V</TableCell>
                  <TableCell>Rear</TableCell>
                  <TableCell>6.0 qt</TableCell>
                  <TableCell className="text-sm text-muted-foreground">Good alternative to F-body pan</TableCell>
                </TableRow>
                <TableRow className="bg-red-50/50 dark:bg-red-900/10">
                  <TableCell className="font-bold text-red-700 dark:text-red-400">Truck / SUV</TableCell>
                  <TableCell>Rear (Deep)</TableCell>
                  <TableCell>6.0 qt</TableCell>
                  <TableCell className="text-sm text-muted-foreground">Stock trucks only - Too deep for car swaps</TableCell>
                </TableRow>
                <TableRow className="bg-blue-50/50 dark:bg-blue-900/10">
                  <TableCell className="font-bold text-blue-700 dark:text-blue-400">GTO / Holden (Front Sump)</TableCell>
                  <TableCell>Front</TableCell>
                  <TableCell>6.0 qt</TableCell>
                  <TableCell className="text-sm text-muted-foreground">240SX, BMW E30, Early Mustang swaps</TableCell>
                </TableRow>
                <TableRow>
                  <TableCell className="font-bold">C6 Corvette</TableCell>
                  <TableCell>Rear (Batwing)</TableCell>
                  <TableCell>5.5 qt</TableCell>
                  <TableCell className="text-sm text-muted-foreground">Good for cornering; wings may interfere with headers</TableCell>
                </TableRow>
                <TableRow>
                  <TableCell className="font-bold">C5 Corvette</TableCell>
                  <TableCell>Rear (Gullwing)</TableCell>
                  <TableCell>5.5 qt</TableCell>
                  <TableCell className="text-sm text-muted-foreground">Designed for high-G cornering; difficult to fit in non-Corvettes</TableCell>
                </TableRow>
                <TableRow>
                  <TableCell className="font-bold">Gen 5 Camaro / G8 / SS</TableCell>
                  <TableCell>Rear</TableCell>
                  <TableCell>8.0 qt</TableCell>
                  <TableCell className="text-sm text-muted-foreground">Late model swaps requiring high oil capacity</TableCell>
                </TableRow>
                <TableRow>
                  <TableCell className="font-bold">LS7 / LS9 Dry Sump</TableCell>
                  <TableCell>Dry Sump</TableCell>
                  <TableCell>External Tank</TableCell>
                  <TableCell className="text-sm text-muted-foreground">Requires external oil tank and lines; longer crank snout</TableCell>
                </TableRow>
              </TableBody>
            </Table>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 gap-6 mt-4">
            <Card className="border-l-4 border-l-green-500">
              <CardHeader className="pb-2">
                <CardTitle className="text-green-700 dark:text-green-400 flex items-center gap-2 text-lg">
                  <CheckCircle2 className="w-5 h-5" />
                  Best for Muscle Car Swaps
                </CardTitle>
              </CardHeader>
              <CardContent className="text-sm space-y-2">
                <p className="font-bold">F-Body Pan (98-02 Camaro/Firebird)</p>
                <p className="text-muted-foreground">
                  The shallow front section makes it ideal for clearing crossmembers in 60s/70s GM muscle cars 
                  (Camaros, Chevelles, Novas). This is the "gold standard" for LS swaps.
                </p>
              </CardContent>
            </Card>

            <Card className="border-l-4 border-l-blue-500">
              <CardHeader className="pb-2">
                <CardTitle className="text-blue-700 dark:text-blue-400 flex items-center gap-2 text-lg">
                  <Info className="w-5 h-5" />
                  When You Need Front Sump
                </CardTitle>
              </CardHeader>
              <CardContent className="text-sm space-y-2">
                <p className="font-bold">GTO / Holden Pan</p>
                <p className="text-muted-foreground">
                  Essential for swaps into chassis that require a front sump location due to steering rack position 
                  (Nissan S-chassis, early Ford Mustangs, BMW E30s).
                </p>
              </CardContent>
            </Card>
          </div>

          <Card className="border border-border/60 bg-muted/20 mt-4">
            <CardContent className="pt-6 flex items-start gap-4">
              <AlertTriangle className="w-6 h-6 text-orange-500 mt-1 shrink-0" />
              <div className="space-y-2">
                <h3 className="font-bold text-base">Common Mistakes</h3>
                <ul className="space-y-2 text-sm text-muted-foreground">
                  <li>• <strong>Using a Truck Pan in a Car:</strong> Truck pans are too deep and will drag on the ground in most car chassis. They almost never fit without modification.</li>
                  <li>• <strong>Ignoring Steering Clearance:</strong> Some pans (like the Gen 5 Camaro) have deep front sections that can hit steering linkage on older cars.</li>
                  <li>• <strong>Corvette "Batwing" Pans:</strong> The massive side wings designed for high-G cornering make them difficult to fit in anything other than a Corvette or wide-rail chassis.</li>
                </ul>
              </div>
            </CardContent>
          </Card>
        </section>

        {/* Visual Cues */}
        <section className="bg-muted/10 p-6 rounded-lg border-2 border-dashed border-border/50">
          <div className="flex items-start gap-4">
             <Info className="w-6 h-6 text-primary mt-1" />
             <div>
               <h3 className="font-bold text-lg mb-2">Visual Spotter's Guide</h3>
               <ul className="space-y-2 text-sm text-muted-foreground">
                 <li>• <strong>Cam Sensor:</strong> Look at the timing cover. If there is a sensor plugged into the front, it is a Gen 4 block.</li>
                 <li>• <strong>Knock Sensors:</strong> Look at the valley cover (under intake). If you see two circular rubber plugs/wires, it is Gen 3. If the valley is flat, it is Gen 4.</li>
                 <li>• <strong>Throttle Body:</strong> 3-bolt is usually Gen 3. 4-bolt (Gold/Silver blade) is Gen 4.</li>
               </ul>
             </div>
          </div>
        </section>
      </div>
    </Layout>
  );
}
