"use strict";
exports.addMonths = addMonths;
var _index = require("./constructFrom.cjs");
var _index2 = require("./toDate.cjs");

var _index3 = require("./_core/getMonth.cjs");
var _index4 = require("./_core/setMonth.cjs");
var _index5 = require("./_core/getDate.cjs");
var _index6 = require("./_core/getFullYear.cjs");
var _index7 = require("./_core/setFullYear.cjs");

/**
 * The {@link addMonths} function options.
 */

/**
 * @name addMonths
 * @category Month Helpers
 * @summary Add the specified number of months to the given date.
 *
 * @description
 * Add the specified number of months to the given date.
 *
 * @typeParam DateType - The `Date` type, the function operates on. Gets inferred from passed arguments. Allows to use extensions like [`UTCDate`](https://github.com/date-fns/utc).
 * @typeParam ResultDate - The result `Date` type, it is the type returned from the context function if it is passed, or inferred from the arguments.
 *
 * @param date - The date to be changed
 * @param amount - The amount of months to be added.
 * @param options - The options object
 *
 * @returns The new date with the months added
 *
 * @example
 * // Add 5 months to 1 September 2014:
 * const result = addMonths(new Date(2014, 8, 1), 5)
 * //=> Sun Feb 01 2015 00:00:00
 *
 * // Add one month to 30 January 2023:
 * const result = addMonths(new Date(2023, 0, 30), 1)
 * //=> Tue Feb 28 2023 00:00:00
 */
function addMonths(date, amount, options) {
  const _date = (0, _index2.toDate)(date, options?.in);
  if (isNaN(amount)) return (0, _index.constructFrom)(options?.in || date, NaN);
  if (!amount) {
    // If 0 months, no-op to avoid changing times in the hour before end of DST
    return _date;
  }
  const dayOfMonth = (0, _index5.getDate)(_date);

  // The JS Date object supports date math by accepting out-of-bounds values for
  // month, day, etc. For example, new Date(2020, 0, 0) returns 31 Dec 2019 and
  // new Date(2020, 13, 1) returns 1 Feb 2021.  This is *almost* the behavior we
  // want except that dates will wrap around the end of a month, meaning that
  // new Date(2020, 13, 31) will return 3 Mar 2021 not 28 Feb 2021 as desired. So
  // we'll default to the end of the desired month by adding 1 to the desired
  // month and using a date of 0 to back up one day to the end of the desired
  // month.
  const endOfDesiredMonth = (0, _index.constructFrom)(
    options?.in || date,
    _date.getTime(),
  );
  (0, _index4.setMonth)(
    endOfDesiredMonth,
    (0, _index3.getMonth)(_date) + amount + 1,
    0,
  );
  const daysInMonth = (0, _index5.getDate)(endOfDesiredMonth);
  if (dayOfMonth >= daysInMonth) {
    // If we're already at the end of the month, then this is the correct date
    // and we're done.
    return endOfDesiredMonth;
  } else {
    // Otherwise, we now know that setting the original day-of-month value won't
    // cause an overflow, so set the desired day-of-month. Note that we can't
    // just set the date of `endOfDesiredMonth` because that object may have had
    // its time changed in the unusual case where where a DST transition was on
    // the last day of the month and its local time was in the hour skipped or
    // repeated next to a DST transition.  So we use `date` instead which is
    // guaranteed to still have the original time.
    (0, _index7.setFullYear)(
      _date,
      (0, _index6.getFullYear)(endOfDesiredMonth),
      (0, _index3.getMonth)(endOfDesiredMonth),
      dayOfMonth,
    );
    return _date;
  }
}
